/* Copyright (c) 2009, Oracle and/or its affiliates.All rights reserved. */

/*

   NAME
     ORAStream.c - sample OraStream usage

   DESCRIPTION
     sample useage of OraStream apis defined in orastruc.h
     OraStreamInit        ---- tested in all the subroutines
     OraStreamOpen        ---- tested in test_open_close()
     OraStreamClose       ---- tested in test_open_close() 
     OraStreamRead        ---- tested in test_read()
     OraStreamReadChar    ---- tested in test_readchar()
     OraStreamWrite       ---- tested in test_write()
     OraStreamWriteChar   ---- tested in test_writechar()
     OraStreamSid         ---- tested in test_osid()
     OraStreamHandle      ---- tested in test_osid()
     OraStreamClone       ---- tested in test_clone()
    

   EXPORT FUNCTION(S)

   INTERNAL FUNCTION(S)

   STATIC FUNCTION(S)
   ORASTREAM_OPEN_F(), ORASTREAM_WRITE_F(), ORASTREAM_CLOSE_F()
   ORASTREAM_READ_F() need to be implemented  

   NOTES

   MODIFIED   (MM/DD/YY)
   zheli      04/29/09 - add standard lib for AIX and SOLARIS
   zheli      02/18/09 - Creation

*/

# include <string.h>
# include <stdlib.h>
# include <stdio.h>

#ifndef ORASTRUC
# include <orastruc.h>
#endif

#ifdef LPX_USAGE
# include <sys/resource.h>
#endif

#ifndef XML_ORACLE
# include <xml.h>
#endif
/*---------------------------------------------------------------------------
                     PRIVATE TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/
int test_open_close();
int test_read();
int test_readchar();
int test_write();
int test_writechar();
int test_osid();
int test_clone();
int test_neg();


oratext * ora_sid1 = "orastream1.xml";
oratext * ora_sid2 = "orastream2.dat";
oratext * ora_sid3 = "ORAStream_w1.out";
oratext * ora_sid4 = "ORAStream_w2.out";
oratext * ora_sid5 = "ORAStream_w3.out";
oratext * ora_sid6 = "ORAStream_w4.out";
oratext * ora_srctext1 = "<client_operation ><txid> sdasdfdsf </txid></client_operation>Streams are byte or character mode: the stream operates in bytmode using OraStreamRead() and OraStreamWriteis provided, the stream operates in character modusing OraStreamReadChar() and OraStreamWriteChar().only complete chars are read/written, and are never split ovebuffer boundaries. ";

static const char *orastream_usage = "usage: ORAStream [options]\n\
options:\n\
      -1            test OraStreamOpen() and OraStreamClose()\n\
      -2            test OraStreamRead()\n\
      -3            test OraStreamReadChar()\n\
      -4            test OraStreamWrite()\n\
      -5            test OraStreamWriteChar()\n\
      -6            Test OraStreamSid() and OraStreamHandle()\n\
      -7            Test OraStreamClone()\n\
      -8            Negative test about OraStreamInit()\n";                                           

                
/*---------------------------------------------------------------------------
                     STATIC FUNCTION DECLARATIONS 
  ---------------------------------------------------------------------------*/
ORASTREAM_WRITE_F(filewrite, sctx, sid, hdl, src, size, written)
{
    FILE *fh = NULL;
    if (fh = (FILE *) hdl->ptr_orastreamhdl)
    {  
        if (!(fwrite(src, sizeof(oratext), size, fh))) 
        {
            printf("Failed to write the data to file %s\n", (oratext*) sid);
            return -1;
        } 
     }
     else 
     {
         printf ("failed to open file %s\n",(oratext *)sid);
         return -1;
     }
    written = (ubig_ora *)size;
    printf("There were %d data written\n", written); 

    return XMLERR_OK;
}

ORASTREAM_OPEN_F(fileopen, sctx, sid, hdl, length)
{
    FILE *fh = NULL;

    printf("Opening orastream %s...\n", (oratext *)sid);

    if (sid && ((fh= fopen(sid, "r")) != NULL))
    {
        printf("Opening orastream %s...\n", (oratext *)sid);
    }
    else
    {  
         printf("Failed to open input file.\n");
         return -1;
     }

    /* store file handle generically, NULL means stdout */
    hdl->ptr_orastreamhdl = fh;

    return XMLERR_OK;
}

ORASTREAM_OPEN_F(fileopen1, sctx, sid, hdl, length)
{
    FILE *fh = NULL;

    printf("Opening orastream %s...\n", (oratext *)sid);

    if (sid && ((fh= fopen(sid, "w")) != NULL))
    {
        printf("Opening orastream %s...\n", (oratext *)sid);
    }
    else
    {
         printf("Failed to open input file.\n");
         return -1;
     }

    /* store file handle generically, NULL means stdout */
    hdl->ptr_orastreamhdl = fh;

    return XMLERR_OK;
}


ORASTREAM_READ_F(fileread, sctx, sid, hdl,
                         dest, size, start, nread, eoi)
{
    FILE *fh = NULL;
    int i =0;
    printf("Reading orastream %s ...\n", (oratext *)sid);

    // read data from file to dest
    if ((fh = (FILE *) hdl->ptr_orastreamhdl) != NULL)
        *nread = fread(dest, 1, size, fh);
    printf("Read %d bytes from orastream...\n", (int) *nread);
    
    *eoi = (*nread < size);
    if (start)
        *start = dest;

    printf("printing document ...\n");
    for(i =0; i < *nread; i++)
    printf("%c", (char)dest[i]);
    printf("\nend ...\n");
    return ORAERR_OK;
}

ORASTREAM_CLOSE_F(fileclose, sctx, sid, hdl)
{
    printf("Closing orastream %s...\n", (oratext*)sid);

    if (hdl->ptr_orastreamhdl)
    {
        fclose((FILE *) hdl->ptr_orastreamhdl);    
        hdl->ptr_orastreamhdl = NULL;
    }
    return XMLERR_OK;
}


/* -------------------------------------------------------------------------
                                     MAIN
   ------------------------------------------------------------------------- */
int main(/* ARGUSED */ int argc, char *argv[])
{
    char ch; 
    
    if ((argc != 2) && (argv[1][0] != '-'))
     {
         printf(orastream_usage);
         return -1;
      } 
     
    ch = argv[1][1];
      switch (ch) 
      {
          case '1':
             test_open_close();
             break; 
          case '2':
              test_read();
              break;
           case '3':
              test_readchar();
              break;
           case '4':
              test_write();
              break;
           case '5':
              test_writechar();
              break;
           case '6':
              test_osid();
              break;
           case '7':
              test_clone();
              break;
           case '8':
              test_neg();
              break;
           default:   
              printf(orastream_usage); 
              break;
            }

    return 0;
}

int test_open_close()
{
    xmlctx *xctx = NULL;
    orastream *ostream = (orastream *) NULL;
    xmlerr ecode = 0; 
    ubig_ora length = 0;
    oraerr oerr = 0 ;
    if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx",(oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
   {
       printf("Failed to create XML context, error %u\n", (unsigned)ecode);
       return -1;
    }

    if (!(ostream = OraStreamInit(NULL, ora_sid1, &oerr,
                                  "open", fileopen,
                                  "close", fileclose,
                                  "read", fileread,
                                  NULL)))
    {
        printf(" Failed to initialize OraStream!, error %u\n", (unsigned)oerr);
        return -1;
    }

    if (OraStreamIsOpen(ostream))
        printf("OraStream is open\n");
    else
        printf("OraStream is closed\n");

    OraStreamOpen(ostream, NULL);
    if (OraStreamIsOpen(ostream))
        printf("OraStream is open\n");
    else
        printf("OraStream is closed\n");

    //open again when orastream is open  
    OraStreamOpen(ostream, NULL);
    if (OraStreamIsOpen(ostream))
        printf("OraStream is open\n");
    else
        printf("OraStream is closed\n");

    OraStreamClose(ostream);
    if (OraStreamIsOpen(ostream))
        printf("OraStream is opend\n");
    else
        printf("OraStream is closed\n");
    
    // open with length = 0
    OraStreamOpen(ostream, &length);
    if (OraStreamIsOpen(ostream))
        printf("OraStream is open\n");
    else
        printf("OraStream is closed\n");

    // open with length = 65535
    OraStreamClose(ostream);
    length = 65535;
    OraStreamOpen(ostream, &length);
    if (!(OraStreamOpen(ostream, &length)));
        if (OraStreamIsOpen(ostream))
            printf("OraStream is open\n");
        else
            printf("OraStream is closed\n");

    //close 
    if(!(OraStreamClose(ostream)))
       if (OraStreamIsOpen(ostream))
           printf("OraStream is open\n");
       else
           printf("OraStream is closed\n");
    
    //term orastream
    if(!(OraStreamTerm(ostream)))
        printf("OraStream was destroyed\n");
     else
        printf("Failed to destroy OraStream\n");
    
 
    XmlDestroy(xctx);

    return 0;

}

int test_read()
{
   xmlctx *xctx = NULL;
   oratext *barray = NULL;
   orastream *ostream = (orastream *) NULL;
   xmlerr ecode = 0;
   ubig_ora  destsize, nread;
   oraerr oerr = 0;
   ub1 eoi = 0;
   nread = destsize = 1024;
   if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
    {
       printf("Failed to create XML context, error %u\n", (unsigned)ecode);
       return -1;
    }

   barray = (oratext *)malloc(sizeof(oratext) * destsize);

   // open function shouled be specified in order to read correctly
   if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                 "open", fileopen,
                                 "read", fileread,
                                 "close", fileclose, 
                                 NULL)))
   {
      printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
      return -1;
   }  
   
   // check readable and writable
    if (OraStreamReadable(ostream))
       printf("ostream is readable\n");
    else
       printf("ostream is not readable\n");

     if (OraStreamWritable(ostream))
       printf("ostream is writable\n");
    else
       printf("ostream is not writable\n");

    if (oerr = OraStreamRead(ostream, barray, destsize, &barray, &nread, &eoi))
    { 
      printf("Failed to read due to orastream was not open, error %u\n", oerr);
    }

   // open orastream
   OraStreamOpen(ostream, NULL);

   // read document 
   OraStreamRead(ostream, barray, destsize, &barray, &nread, &eoi);

   //
   OraStreamTerm(ostream);
   free(barray);
   XmlDestroy(xctx);
   return 0;
}


int test_readchar()
{
   xmlctx *xctx = NULL;
   oratext *barray = NULL;
   orastream *ostream = (orastream *) NULL;
   xmlerr ecode = 0;
   ubig_ora  destsize, nread;
   oraerr oerr = 0;
   ub1 eoi = 0;
   nread = destsize = 1024;
   if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
    {
       printf("Failed to create XML context, error %u\n", (unsigned)ecode);
       return -1;
    }

   barray = (oratext *)malloc(sizeof(oratext) * destsize);

   // open function shouled be specified in order to read correctly
   if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                 "open", fileopen,
                                 "read_char", fileread,
                                 "close", fileclose,
                                 NULL)))
   {
      printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
      return -1;
   }

   // check readable and writable
    if (OraStreamReadable(ostream))
       printf("ostream is readable\n");
    else
       printf("ostream is not readable\n");

     if (OraStreamWritable(ostream))
       printf("ostream is writable\n");
    else
       printf("ostream is not writable\n");

    oerr = OraStreamReadChar(ostream, barray, destsize, &barray, &nread, &eoi);
    if (oerr)
    {
      printf("Failed to read due to orastream was not open, error %u\n", oerr);
    }
   
   // open orastream
   OraStreamOpen(ostream, NULL);
   
   // read document 
   OraStreamReadChar(ostream, barray, destsize, &barray, &nread, &eoi);
    
   //
   OraStreamTerm(ostream);
   free(barray);
   XmlDestroy(xctx);
   return 0;
}
    

int test_write()
{  
   xmlctx *xctx = NULL; 
   orastream *ostream = (orastream *)NULL;
   xmlerr ecode = 0;
   ubig_ora  nwrote, destsize;
   oraerr oerr = 0;
   nwrote = destsize = 1024; 

   if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
    {
       printf("Failed to create XML context, error %u\n", (unsigned)ecode);
       return -1;
    }

   if (!(ostream = OraStreamInit(NULL, ora_sid3, &oerr,
                                 "open", fileopen1,
                                 "write", filewrite,
                                 "close", fileclose,
                                 NULL)))
   {
      printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
      return -1;
   }

   // check readable and writable
    if (OraStreamReadable(ostream))
       printf("ostream is readable\n");
    else
       printf("ostream is not readable\n");

     if (OraStreamWritable(ostream))
       printf("ostream is writable\n");
    else
       printf("ostream is not writable\n");
    // open orastream
    OraStreamOpen(ostream, NULL);
   
   // write document
   OraStreamWrite(ostream, ora_srctext1, strlen((char*)ora_srctext1), &nwrote);
    
   // close orastream
   OraStreamClose(ostream); 
  
   // term orastream
   OraStreamTerm(ostream);

   XmlDestroy(xctx);
   return 0;
}


int test_writechar()
{   
   xmlctx *xctx = NULL;
   orastream *ostream = (orastream *)NULL;
   xmlerr ecode = 0;
   ubig_ora  nwrote, destsize;
   oraerr oerr = 0;
   nwrote = destsize = 1024;
 
   if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
    {
   printf("Failed to create XML context, error %u\n", (unsigned)ecode);
       return -1;
    }

   if (!(ostream = OraStreamInit(NULL, ora_sid4, &oerr,
                                 "open", fileopen1,    
                                 "write_char", filewrite,
                                 "close", fileclose,
                                 NULL)))
   {
      printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
      return -1;
   }

   // check readable and writable
    if (OraStreamReadable(ostream))
       printf("ostream is readable\n");
    else
       printf("ostream is not readable\n");

     if (OraStreamWritable(ostream))
       printf("ostream is writable\n");
    else
       printf("ostream is not writable\n");


    // open orastream
    OraStreamOpen(ostream, NULL);
   
   // write document 
   OraStreamWriteChar(ostream, ora_srctext1, strlen((char *)ora_srctext1), &nwrote);
    
   // close orastream
   OraStreamClose(ostream);
   
   // term orastream  
   OraStreamTerm(ostream);

   XmlDestroy(xctx);
   return 0;
}
    



int test_osid() 
{   
    xmlctx *xctx = NULL;
    oratext *barray = NULL;
    orastream  *ostream = (orastream *) NULL;
    xmlerr ecode = 0;
    orastreamhdl *ohdl = (orastreamhdl *)NULL;
    ubig_ora  destsize, nread;
    oraerr oerr = 0;
    ub1 eoi = 0;
    FILE *fh = (FILE *)NULL;
    int i;
    nread = destsize = 1024;
    if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0,
                      (oratext *)"US-ASCII", NULL)))
    {
        printf("Failed to create XML context, error %u\n", (unsigned)ecode);
        return -1;
    }

    barray = (oratext *)malloc(sizeof(oratext) * destsize);

    // open function shouled be specified in order to read correctly
    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read", fileread,
                                  "close", fileclose,
                                  NULL)))
    {
        printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
        return -1;
    }
    
    
    /* set OraStream Id after OraStreamOpen, the file handles for the OraStream
     * will not change immediately, for the file handle is set in function 
     * fileopen, close and reopen the orastream will work. 
     */
    // open stream
    OraStreamOpen(ostream, NULL);
    // set OraStream id = ora_sid2
    OraStreamSid(ostream, ora_sid2, NULL);
    // 
    ohdl = OraStreamHandle(ostream);
    // get file handle from orastream, and print it
    fh = (FILE *)ohdl->ptr_orastreamhdl;
    nread = fread(barray, 1, destsize, fh);
    // print data read from file
    printf("printing document ...\n");
    for(i =0; i < nread; i++)
    printf("%c", (char)barray[i]);
    printf("\nend ...\n");

    // close
    OraStreamClose(ostream);
    OraStreamOpen(ostream, NULL);
    // get file handle from orastream, and print it
    ohdl = OraStreamHandle(ostream);
    fh = (FILE *)ORASTREAM_PTR(ohdl);
    nread = fread(barray, 1, destsize, fh);

    // print data read from file
    printf("printing document ...\n");
    for(i =0; i < nread; i++)
    printf("%c", (char)barray[i]);
    printf("\nend ...\n");
    // close OraStream
    OraStreamClose(ostream);
   
 
    //set OraStreamSid 
    OraStreamSid(ostream, ora_sid1, NULL);
    // open orastream
    OraStreamOpen(ostream, NULL);
    //read orastream
    OraStreamRead(ostream, barray, destsize, &barray, &nread, &eoi);
    // close OraStream
    OraStreamClose(ostream);
    
    // set OraStreamSid with new sid and original osid
    OraStreamSid(ostream, ora_sid2, (void*)&ora_sid1);
    // open orastream
    OraStreamOpen(ostream, NULL);
    //read orastream
    OraStreamRead(ostream, barray, destsize, &barray, &nread, &eoi);
    // close OraStream
    OraStreamClose(ostream);
    OraStreamTerm(ostream);
   
    free(barray);
    XmlDestroy(xctx);
    return 0;
}


int test_clone()
{
    xmlctx *xctx = NULL;
    oratext *barray = NULL;
    orastream *clostream = (orastream *)NULL, *ostream = (orastream *) NULL;
    xmlerr ecode = 0;
    oramemctx *memctx = NULL;
    ubig_ora  destsize, nread, nwrote;
    oraerr oerr = 0;
    ub1 eoi = 0;
    int i;

    nread = nwrote = destsize = 1024;
    /* initialize myctx here */
    memctx = OraMemInit(NULL, NULL, NULL);

    if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0, (oratext *)"US-ASCII", "memory_context", memctx, NULL)))
    {
        printf("Failed to create XML context, error %u\n", (unsigned)ecode);
        return -1;
    }
    
    barray = (oratext *)malloc(sizeof(oratext) * destsize);
    
    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read", fileread,
                                  "close", fileclose,
                                  NULL)))
    {
       printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
       return -1;
    }

    /* OraStreamClone doesn't clone sid from the original orastream
     * it just clone function related to the orastream 
     * set it now or set it by OraStreamSid later
     */

    if (!(clostream = OraStreamClone(memctx, ora_sid2, ostream, &oerr)))
    {
        printf("clone orastream failed,  error %u\n", oerr);
    }

    // check readable and writable
    if (OraStreamReadable(clostream))
        printf("clostream is readable\n");
    else
        printf("clostream is not readable\n");
    if (OraStreamWritable(clostream))
        printf("clostream is writable\n");
    else
        printf("clostream is not writable\n");

    OraStreamOpen(clostream, NULL);
    OraStreamRead(clostream, barray, destsize, &barray, &nread, &eoi);

    OraStreamOpen(ostream, NULL);
    OraStreamRead(ostream, barray, destsize, &barray, &nread, &eoi);

    OraStreamClose(ostream);
    OraStreamClose(clostream);  
    OraStreamTerm(ostream);
    OraStreamTerm(clostream);
    
    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read_char", fileread,
                                  "close", fileclose,
                                  NULL)))
    {
       printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
       return -1;
    }

    if (!(clostream = OraStreamClone(memctx, NULL, ostream, &oerr)))
    {
        printf("clone orastream failed,  error %u\n", oerr);
    }

    // check readable and writable
    if (OraStreamReadable(clostream))
        printf("clostream is readable\n");
    else
        printf("clostream is not readable\n");

    if (OraStreamWritable(clostream))
        printf("clostream is writable\n");
    else
        printf("clostream is not writable\n");

    OraStreamSid(clostream, ora_sid2, NULL);
    OraStreamOpen(clostream, NULL);
    OraStreamReadChar(clostream, barray, destsize, &barray, &nread, &eoi);
    //close orastream
    OraStreamClose(ostream);
    OraStreamClose(clostream);
    // term orastream 
    OraStreamTerm(ostream);
    OraStreamTerm(clostream);

    
    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                 "open", fileopen1,
                                 "write", filewrite,
                                 "close", fileclose,
                                 NULL)))
    {
        printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
        return -1;
    }
    

    if (!(clostream = OraStreamClone(memctx,ora_sid5, ostream, &oerr)))
    {
        printf("clone orastream failed,  error %u\n", oerr);
    }

    // check readable and writable
    if (OraStreamReadable(clostream))
        printf("clostream is readable\n");
    else
        printf("clostream is not readable\n");

    if (OraStreamWritable(clostream))
        printf("clostream is writable\n");
    else
        printf("clostream is not writable\n");

    // open orastream
    OraStreamOpen(clostream, NULL);

    // write document 
    OraStreamWrite(clostream, ora_srctext1, strlen((char*)ora_srctext1), &nwrote);

    //close orastream
    OraStreamClose(ostream);
    OraStreamClose(clostream);
    // term orastream
    OraStreamTerm(ostream);
    OraStreamTerm(clostream);


     if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen1,
                                  "write_char", filewrite,
                                  "close", fileclose,
                                  NULL)))
    {
        printf("Failed to initialize OrsStream, error %u\n",(unsigned)oerr);
        return -1;
    }


    if (!(clostream = OraStreamClone(memctx,ora_sid6, ostream, &oerr)))
    {
        printf("clone orastream failed,  error %u\n", oerr);
    }


    // check readable and writable
    if (OraStreamReadable(clostream))
        printf("clostream is readable\n");
    else
        printf("clostream is not readable\n");

    if (OraStreamWritable(clostream))
        printf("clostream is writable\n");
    else
        printf("clostream is not writable\n");

    // open orastream
    OraStreamOpen(clostream, NULL);

    // write document 
    OraStreamWriteChar(clostream, ora_srctext1, strlen((char*)ora_srctext1), &nwrote);

    // close orastream
    OraStreamClose(ostream);
    OraStreamClose(clostream);
    //term orastream 
    OraStreamTerm(ostream);
    OraStreamTerm(clostream);

    free(barray);
    OraMemFree(memctx, NULL);
    OraMemTerm(memctx);
    XmlDestroy(xctx);
    return 0;
    
}

/* negative test for OraStreamInit() */
int test_neg()
{   
    xmlctx *xctx = NULL;
    orastream  *ostream = (orastream *)NULL;
    xmlerr ecode = 0;
    oraerr oerr = 0;
    ub1 eoi = 0;
    if (!(xctx = XmlCreateNew(&ecode, (oratext *)"stream_xctx", (oratext **)0, 0, (oratext *)"US-ASCII", NULL)))
    {
        printf("Failed to create XML context, error %u\n", (unsigned)ecode);
        return -1;
    }


    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read", fileread,
                                  "read_char", fileread,
                                  "close", fileclose,
                                   NULL)))
    {
        printf("Failed to initialize OrsStream with read and read_char, error %u\n",(unsigned)oerr);
    }

    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read", fileread,
                                  "write", filewrite,
                                  "close", fileclose,
                                  NULL)))
    {
       printf("Failed to initialize OrsStream with read and write, error %u\n",(unsigned)oerr);
    }

/* disable this  */
/*    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read", fileread,
                                  "write_char", filewrite,
                                  "close", fileclose,
                                  NULL)))
      {
        printf("Failed to initialize OrsStream with read and write_char, error %u\n",(unsigned)oerr);
       }

    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read_char", fileread,
                                  "write", filewrite,
                                  "close", fileclose,
                                  NULL)))
    {
        printf("Failed to initialize OrsStream with read_char and write, error %u\n",(unsigned)oerr);
    }
*/
    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "read_char", fileread,
                                  "write_char", filewrite,
                                  "close", fileclose,
                                  NULL)))
    {
       printf("Failed to initialize OrsStream with read_char and write_char, error %u\n",(unsigned)oerr);
    }


    if (!(ostream = OraStreamInit(NULL,ora_sid1, &oerr,
                                  "open", fileopen,
                                  "cool", fileread,
                                  "hot", filewrite,
                                  "close", fileclose,
                                  NULL)))
    {
       printf("Failed to initialize OrsStream with cool and hot, error %u\n",(unsigned)oerr);
    }

   return 0;

}  

/* end of file ORAStream.c */
